#region usings
using System;
using System.IO;
using System.ComponentModel.Composition;

using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;
#endregion usings

namespace VVVV.Nodes
{
	#region PluginInfo
	[PluginInfo(Name = "CRC_Calculator", Category = "Raw", Version = "1", Help = "Basic raw template which copies up to count bytes from the input to the output", Tags = "")]
	#endregion PluginInfo
	public class C1RawCRC_CalculatorNode : IPluginEvaluate, IPartImportsSatisfiedNotification
	{
		#region fields & pins
		[Input("Input")]
		public ISpread<Stream> FStreamIn;

		[Output("Output")]
		public ISpread<Stream> FStreamOut;
		
		//when dealing with byte streams (what we call Raw in the GUI) it's always
		//good to have a byte buffer around. we'll use it when copying the data.
		readonly byte[] FBuffer = new byte[1024];
		byte[] crcBuffer = new byte[2];
		#endregion fields & pins

		//called when all inputs and outputs defined above are assigned from the host
		public void OnImportsSatisfied()
		{
			//start with an 1 stream output
			FStreamOut.SliceCount = 0;
			
		}

		//called when data for any output pin is requested
		public void Evaluate(int spreadMax)
		{
			//ResizeAndDispose will adjust the spread length and thereby call
			//the given constructor function for new slices and Dispose on old
			//slices.
			FStreamOut.ResizeAndDispose(spreadMax, () => new MemoryStream());
			
			for (int i = 0; i < spreadMax; i++) {
				//get the input stream
				var inputStream = FStreamIn[i];
				//get the output stream (this works because of ResizeAndDispose above)
				var outputStream = FStreamOut[i];
				//set the length of the output stream to 2 bytes (crc16 modbus)
				outputStream.SetLength(2);
				
				//reset the positions of the streams
				inputStream.Position = 0;
				outputStream.Position = 0;
				var count = (int)inputStream.Length;
				
				inputStream.Read(FBuffer, 0,count);
	
				crcBuffer = BitConverter.GetBytes(ModRTU_CRC(FBuffer, count));
				
				outputStream.Write(crcBuffer, 0, 2);
			}	
			//this will force the changed flag of the output pin to be set
			FStreamOut.Flush(true);
		}
		
		// Compute the MODBUS RTU CRC
		UInt16 ModRTU_CRC(byte[] buf, int len)
		{
		  UInt16 crc = 0xFFFF;
		  
		  for (int pos = 0; pos < len; pos++) {
		    crc ^= (UInt16)buf[pos];          // XOR byte into least sig. byte of crc
		  
		    for (int i = 8; i != 0; i--) {    // Loop over each bit
		      if ((crc & 0x0001) != 0) {      // If the LSB is set
		        crc >>= 1;                    // Shift right and XOR 0xA001
		        crc ^= 0xA001;
		      }
		      else                            // Else LSB is not set
		        crc >>= 1;                    // Just shift right
		    }
		  }
		  // Note, this number has low and high bytes swapped, so use it accordingly (or swap bytes)
		  return crc;  
		}
		
	}
}
